#include "gstream.h"
#include <cstring>
#include <cstdlib>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <errno.h>
#include <streambuf>
#include <iostream>
#include <string>
#include <sstream>
#include <vector>


namespace std
{


static string vo="";
static vector<string> dest;
static string tmpdir="";
static int debug=0;


void gstream_debug(const int level)
{
    debug=level;
}


int gstream_debug()
{
    return debug;
}


static void ginit()
{
    if ( getenv("LCG_CATALOG_TYPE")==NULL )
    {
	cerr << "[gstream] Environmental variable LCG_CATALOG_TYPE is not set!\n[gstream]\tvoid ginit()\n";
	::abort();
    }
    if ( getenv("LFC_HOST")==NULL )
    {
	cerr << "[gstream] Environmental variable LFC_HOST is not set!\n[gstream]\tvoid ginit()\n";
	::abort();
    }
    char *vo_cstr=getenv("LCG_GFAL_VO");
    if ( vo_cstr==NULL )
    {
	cerr << "[gstream] Environmental variable LCG_GFAL_VO is not set!\n[gstream]\tvoid ginit()\n";
	::abort();
    }
    vo=string(vo_cstr);
    char *dests_cstr=getenv("DEST");
    if ( dests_cstr==NULL )
    {
	cerr << "[gstream] Environmental variable DEST is not set!\n[gstream]\tvoid ginit()\n";
	::abort();
    }
    string dests=string(dests_cstr);
    for ( unsigned int i=0 ; i<dests.length() ; ++i ) if ( dests[i]==',' ) dests[i]=' ';
    istringstream iss;
    string namebuff;
    iss.str(dests);
    dest.resize(0);
    while ( (iss>>namebuff) ) dest.push_back(namebuff);
    iss.str("");
    iss.clear();
    ostringstream command;
    int err;
    command << "grid-proxy-info -e >&/dev/null";
    err=system(command.str().c_str());
    command.str("");
    command.clear();
    if ( err!=0 )
    {
	cerr << "[gstream] No valid grid-proxy!\n[gstream]\tvoid ginit()\n";
	::abort();
    }
    char *tmpdir_cstr=getenv("TMPDIR");
    if ( tmpdir_cstr==NULL ) tmpdir=""; else tmpdir=string(tmpdir_cstr);
    if ( tmpdir=="" )
    {
	string pwd="";
	char *pwd_cstr=getenv("PWD");
	if ( pwd_cstr==NULL ) pwd=""; else pwd=string(pwd_cstr);
	tmpdir=pwd;
    }
}


static void cleanup(const string &filename, const string &directoryname)
{
    ostringstream command;
    command << "rm -f " << filename;
    system(command.str().c_str());
    command.str("");
    command.clear();
    command << "rm -rf " << directoryname;
    system(command.str().c_str());
    command.str("");
    command.clear();
}


static void mkstagefile(const string &gfilename, string &filename, string &directoryname)
{
    string fname;
    istringstream iss(gfilename);
    while ( getline(iss, fname, '/') ) { }
    iss.str("");
    iss.clear();
    char uniquedirtag[]="gstream_XXXXXX";
    directoryname=tmpdir+"/"+uniquedirtag;
    char *uniquedir=new char[directoryname.length()+16];
    sprintf(uniquedir, "%s", directoryname.c_str());
    if ( mkdtemp(uniquedir)==NULL )
    {
	cerr << "[gstream] Could not create temporary directory!\n[gstream]\tvoid mkstagefile(const string&, string&, string&)\n";
	::abort();
    }
    string dname;
    iss.str(uniquedir);
    while ( getline(iss, dname, '/') ) { }
    iss.str("");
    iss.clear();
    delete[] uniquedir;
    directoryname=tmpdir+"/"+dname;
    filename=directoryname+"/"+fname;
}


static bool gexists(const string &gfilename)
{
    ostringstream command;
    command << "lcg-lg --vo " << vo << " ";
    command << "lfn:" << gfilename << " >&/dev/null";
    int err=system(command.str().c_str());
    command.str("");
    command.clear();
    return (err==0);
}


static int gget(const string &gfilename, const string &filename, const string &directoryname)
{
    int errorCode=0;
    if ( debug>=1 ) cerr << "[gstream] Getting: " << gfilename << endl;
    ostringstream command;
    command << "lcg-cp --vo " << vo << " ";
    command << "lfn:" << gfilename << " file:" << filename;
    if ( debug>=2 ) cerr << "[gstream] Command:\n" << command.str() << endl;
    int err=1;
    int tries=0;
    while ( err!=0 )
    {
	err=system(command.str().c_str());
	++tries;
	if ( tries>=5 )
	{
	    cerr << "[gstream] Error getting file " << gfilename << " . Giving up after the 5-th try!\n[gstream]\tvoid gget(const string&, const string&, const string&)\n";
	    cleanup(filename, directoryname);
	    errorCode=-1;
	    break;
	}
	if ( err!=0 ) sleep(10);
    }
    command.str("");
    command.clear();
    return errorCode;
}


static int gdel(const string &gfilename)
{
    int errorCode=0;
    if ( debug>=1 ) cerr << "[gstream] Deleting: " << gfilename << endl;
    ostringstream command;
    command << "lcg-del --vo " << vo << " ";
    command << "-a ";
    command << "lfn:" << gfilename;
    if ( debug>=2 ) cerr << "[gstream] Command:\n" << command.str() << endl;
    int err=1;
    int tries=0;
    while ( err!=0 )
    {
	err=system(command.str().c_str());
	++tries;
	if ( tries>=5 )
	{
	    cerr << "[gstream] Error deleting file " << gfilename << " . Giving up after the 5-th try!\n[gstream]\tvoid gdel(const string&)\n";
	    errorCode=-1;
	    break;
	}
	if ( err!=0 ) sleep(10);
    }
    command.str("");
    command.clear();
    return errorCode;
}


static int gput(const string &gfilename, const string &filename, const string &directoryname)
{
    int errorCode=0;
    if ( debug>=1 ) cerr << "[gstream] Putting: " << gfilename << endl;
    ostringstream command;
    command << "lcg-cr --vo " << vo << " ";
    command << "-d " << dest[0] << " ";
    command << "-l lfn:" << gfilename << " file:" << filename << " 1>/dev/null";
    if ( debug>=2 ) cerr << "[gstream] Command:\n" << command.str() << endl;
    int err=1;
    int tries=0;
    while ( err!=0 )
    {
	err=system(command.str().c_str());
	++tries;
	if ( tries>=5 )
	{
	    cerr << "[gstream] Error putting file " << gfilename << " . Giving up after the 5-th try!\n[gstream]\tvoid gput(const string&, const string&, const string&)\n";
	    cleanup(filename, directoryname);
	    errorCode=-1;
	    break;
	}
	if ( err!=0 ) sleep(10);
    }
    command.str("");
    command.clear();
    return errorCode;
}


static int grep(const string &gfilename)
{
    int errorCode=0;
    for ( unsigned int i=1 ; i<dest.size() ; ++i )
    {
	if ( i==1 && debug>=1 ) cerr << "[gstream] Replicating: " << gfilename << endl;
	ostringstream command;
	command << "lcg-rep --vo " << vo << " ";
	command << "-d " << dest[i] << " ";
	command << "lfn:" << gfilename << " 1>/dev/null";
	if ( debug>=2 ) cerr << "[gstream] Command:\n" << command.str() << endl;
	int err=1;
	int tries=0;
	while ( err!=0 )
	{
	    err=system(command.str().c_str());
	    ++tries;
	    if ( tries>=5 )
	    {
		cerr << "[gstream] Error replicating file " << gfilename << " . Giving up after the 5-th try!\n[gstream]\tvoid grep(const string&)\n";
		errorCode=-1;
		break;
	    }
	    if ( err!=0 ) sleep(10);
	}
	command.str("");
	command.clear();
    }
    return errorCode;
}


gstream::gstream()
 : opened_(false)
{
}


gstream::gstream(const string &name, ios::openmode mode)
 : opened_(false)
{
    gstream::open(name, mode);
}


gstream::~gstream()
{
    gstream::close();
}


gstream& gstream::open(const string &name, ios::openmode mode)
{
    if ( opened_==false )
    {
	gfilename_=name;
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    ginit();
	    mkstagefile(gfilename_, filename_, directoryname_);
	    if ( gexists(gfilename_) ) gget(gfilename_, filename_, directoryname_);
	}
	else 
	{
	    filename_=gfilename_;
	    directoryname_="";
	}
	fstream::open(filename_.c_str(), mode);
	opened_=true;
    }
    return *this;
}


gstream& gstream::close()
{
    if ( opened_==true )
    {
	fstream::close();
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    if ( gexists(gfilename_) ) gdel(gfilename_);
	    gput(gfilename_, filename_, directoryname_);
	    grep(gfilename_);
	    cleanup(filename_, directoryname_);
	}
	opened_=false;
    }
    return *this;
}


igstream::igstream()
 : opened_(false)
{
}


igstream::igstream(const string &name, ios::openmode mode)
 : opened_(false)
{
    igstream::open(name, mode);
}


igstream::~igstream()
{
    igstream::close();
}


igstream& igstream::open(const string &name, ios::openmode mode)
{
    if ( opened_==false )
    {
	gfilename_=name;
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    ginit();
	    if ( gexists(gfilename_) )
	    {
		mkstagefile(gfilename_, filename_, directoryname_);
		gget(gfilename_, filename_, directoryname_);
	    }
	    else
	    {
		cerr << "[gstream] File " << gfilename_ << " does not exist.\n[gstream]\tigstream& igstream::open(const string&, ios::openmode)\n";
	    }
	}
	else
	{
	    filename_=gfilename_;
	    directoryname_="";
	}
	ifstream::open(filename_.c_str(), mode);
	opened_=true;
    }
    return *this;
}


igstream& igstream::close()
{
    if ( opened_==true )
    {
	ifstream::close();
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid ) cleanup(filename_, directoryname_);
	opened_=false;
    }
    return *this;
}


ogstream::ogstream()
 : opened_(false)
{
}


ogstream::ogstream(const string &name, ios::openmode mode)
 : opened_(false)
{
    ogstream::open(name, mode);
}


ogstream::~ogstream()
{
    ogstream::close();
}


ogstream& ogstream::open(const string &name, ios::openmode mode)
{
    if ( opened_==false )
    {
	gfilename_=name;
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    ginit();
	    mkstagefile(gfilename_, filename_, directoryname_);
	    if ( (mode&ios::app!=0) && gexists(gfilename_) )
	    {
		gget(gfilename_, filename_, directoryname_);
	    }
	}
	else
	{
	    filename_=gfilename_;
	    directoryname_="";
	}
	ofstream::open(filename_.c_str(), mode);
	opened_=true;
    }
    return *this;
}


ogstream& ogstream::close()
{
    if ( opened_==true )
    {
	ofstream::close();
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    if ( gexists(gfilename_) ) gdel(gfilename_);
	    gput(gfilename_, filename_, directoryname_);
	    grep(gfilename_);
	    cleanup(filename_, directoryname_);
	}
	opened_=false;
    }
    return *this;
}



// ================= gpstream stuff ====================================


static void searchandreplace(string &str, const string &f, const string &r)
{
    unsigned int pos=str.find(f);
    while ( pos<str.length() ) { str.replace(pos, f.length(), r); pos=str.find(f); }
}


igpstream::igpstream()
 : opened_(false)
{
}


igpstream::igpstream(const string &name, const string &pipe)
 : opened_(false)
{
    igpstream::open(name, pipe);
}


igpstream::~igpstream()
{
    igpstream::close();
}


igpstream& igpstream::open(const string &name, const string &pipe)
{
    if ( opened_==false )
    {
	gfilename_=name;
	pipename_=pipe;
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    ginit();
	    if ( gexists(gfilename_) )
	    {
		mkstagefile(gfilename_, filename_, directoryname_);
		gget(gfilename_, filename_, directoryname_);
	    }
	    else
	    {
		cerr << "[gstream] File " << gfilename_ << " does not exist.\n[gstream]\tigpstream& igpstream::open(const string&, ios::openmode)\n";
	    }
	}
	else
	{
	    filename_=gfilename_;
	    directoryname_="";
	}
	searchandreplace(pipename_, "%f", filename_);
	ipstream::open(pipename_);
	opened_=true;
    }
    return *this;
}


igpstream& igpstream::close()
{
    if ( opened_==true )
    {
	ipstream::close();
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid ) cleanup(filename_, directoryname_);
	opened_=false;
    }
    return *this;
}


ogpstream::ogpstream()
 : opened_(false)
{
}


ogpstream::ogpstream(const string &name, const string &pipe)
 : opened_(false)
{
    ogpstream::open(name, pipe);
}


ogpstream::~ogpstream()
{
    ogpstream::close();
}


ogpstream& ogpstream::open(const string &name, const string &pipe)
{
    if ( opened_==false )
    {
	gfilename_=name;
	pipename_=pipe;
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    ginit();
	    mkstagefile(gfilename_, filename_, directoryname_);
	    if ( gexists(gfilename_) )
		gget(gfilename_, filename_, directoryname_);
	}
	else
	{
	    filename_=gfilename_;
	    directoryname_="";
	}
	searchandreplace(pipename_, "%f", filename_);
	opstream::open(pipename_);
	opened_=true;
    }
    return *this;
}


ogpstream& ogpstream::close()
{
    if ( opened_==true )
    {
	opstream::close();
	bool isongrid=false;
	if ( gfilename_.length()>=6 )
	{
	    if ( gfilename_.substr(0, 6)=="/grid/" ) isongrid=true;
	}
	if ( isongrid )
	{
	    if ( gexists(gfilename_) ) gdel(gfilename_);
	    gput(gfilename_, filename_, directoryname_);
	    grep(gfilename_);
	    cleanup(filename_, directoryname_);
	}
	opened_=false;
    }
    return *this;
}


}

