#include "Utilities/Configuration/interface/Architecture.h"

#include "TrackerReco/ClusterShaper/interface/FitStripCluster.h"

#include "TrackerReco/ClusterShaper/interface/Transformations.h"
#include "TrackerReco/ClusterShaper/interface/LevenbergMarquardt.h"

#include <iostream>

inline double Coupling() { return 0.12; }

extern geom_t geom;

/*****************************************************************************/
FitStripCluster::FitStripCluster() { }

/*****************************************************************************/
FitStripCluster::~FitStripCluster() { }

/*****************************************************************************/
double FitStripCluster::function(double z, int rank)
{
 // Gauss
 if(rank == 0) return(2 * z*z/2);
 if(rank == 1) return(    z  );
          else return(    1. );
}

/*****************************************************************************/
int FitStripCluster::getPath(double point[], vector<pixel_t>* pixel)
{
 int n;

 // increasing order! +2 coupling
 n = (int)point[1] - (int)point[0] + 1 + 2;

 // Calculate position and path length
 for(int i=0; i<n; i++)
 {
  pixel_t pix;

  pix.position[0] = (int)point[0] + i - 1;
  if(i==0 || i==1 || i==n-2 || i==n-1)
  {
    if(i==0  ) pix.normal[0][0] = -   Coupling(); 
    if(i==1  ) pix.normal[0][0] = -(1-Coupling()); 
    if(i==n-2) pix.normal[0][0] =  (1-Coupling());
    if(i==n-1) pix.normal[0][0] =     Coupling();

    pix.endpoint = 1;
  }
  else
  { pix.endpoint = 0; pix.normal[0][0] = 0.; }

  pix.length = 1.;
  if(i==0  ) pix.length = 0.;
  if(i==1  ) pix.length = (int)point[0]+1 - point[0];
  if(i==n-2) pix.length = point[1] - (int)point[1];
  if(i==n-1) pix.length = 0.;

  pixel->push_back(pix);
 }

 // Add coupling
 vector<double> length(n);

 for(int i=0; i<n; i++)
  length[i] = (i>0   ? (*pixel)[i-1].length *        Coupling() : 0) +
                       (*pixel)[i  ].length * (1 - 2*Coupling()) +
	      (i<n-1 ? (*pixel)[i+1].length *        Coupling() : 0); 

 for(int i=0; i<n; i++)
  (*pixel)[i].length = length[i];

 return(n);
}

/*****************************************************************************/
void FitStripCluster::negativeDistance(pixel_t *pixel, double endpoint[])
{
 pixel->sign   = -1;
 pixel->length =  1.;
}

/*****************************************************************************/
void FitStripCluster::fillAdc
  (int nchannel,channel_t *channel,
   int *npixel ,vector<pixel_t>* pixel, double endpoint[])
{
 // Clear adc
 for(int i=0; i<*npixel; i++)
 {
  (*pixel)[i].adc  = 0;
  (*pixel)[i].sign = 1;
 }

 // Try to match channels
 for(int j=0; j<nchannel; j++)
 {
  int found = 0;

  for(int i=0; i<*npixel; i++)
  {
   if(channel[j].position[0] == (*pixel)[i].position[0])
   { (*pixel)[i].adc = channel[j].adc; found=1; }
  }

  if(!found) // This channel is not touched by the line, a new one, add it!
  {
   pixel_t pix;
   pix.position[0] = channel[j].position[0];
   pix.adc         = channel[j].adc;

   // Calculate negative distance
   negativeDistance(&pix, endpoint);
   pixel->push_back(pix);
   
   (*npixel)++;
  }
 }
}

/*****************************************************************************/
void FitStripCluster::prepareStrips(double endpoint[])
{
 // Reset strips
 pixel.clear();

 // Calculate path
 npixel = getPath(endpoint, &pixel);

 // Fill adcs
 fillAdc(nchannel,&channel[0], &npixel,&pixel, endpoint);
}

/*****************************************************************************/
void FitStripCluster::getFunction(double cluster[], double *chi2)
{
 // Get cluster endpoints
 double endpoint[2];
 endpoint[0] = cluster[0] - cluster[1];
 endpoint[1] = cluster[0] + cluster[1];

 prepareStrips(endpoint);

 // Clear
 *chi2 = 0.;

 for(int i=0; i<npixel; i++)
 {
  double val = cluster[2] * pixel[i].sign * pixel[i].length;
  *chi2 += function(pixel[i].adc - val, 0);
 }
}

/*****************************************************************************/
void FitStripCluster::collectDerivativesCluster
(double par[],pixel_t *pixel, double length,
 double der[], double *val)
{
 // center
 der[0] = par[2] * pixel->sign * pixel->normal[0][0];
 
 // half-length
 der[1] = par[2] * pixel->endpoint;
 
 // energy loss
 der[2] = pixel->sign * pixel->length;

 // value
 *val = par[2] * pixel->sign * pixel->length;
}

/*****************************************************************************/
void FitStripCluster::getDerivatives
  (double cluster[], double beta[],Matrix<double>* alpha)
{
 double der[Spar],val, chi2=0.;

 double endpoint[2];
 endpoint[0] = cluster[0] - cluster[1];
 endpoint[1] = cluster[0] + cluster[1];

 // Clear
 for(int k=0; k<Spar; k++)
 {
  beta[k] = 0.;
  for(int l=0; l<Spar; l++) (*alpha)(k,l) = 0.;
 }

 double length = endpoint[1] - endpoint[0];

 // For all the pixels
 for(int i=0; i<npixel; i++)
 {
  // Get derivatives
  collectDerivativesCluster(cluster,&pixel[i], length, der,&val);

  chi2 += function(pixel[i].adc - val, 0);

  // Fill beta and alpha (for cluster)
  for(int k=0; k<Spar; k++)
  {
   beta[k] += function(pixel[i].adc -  val, 1) * der[k];
   for(int l=0; l<Spar; l++)
    (*alpha)(k,l) += function(pixel[i].adc -  val, 2) * der[k] * der[l];
  }
 }
}

/*****************************************************************************/
void FitStripCluster::fit
  (int ncha,int val[][3], double endpoint[], double track[][Npar+1])
{
 // Allocate
 channel.reserve(10);

 // Copy
 channel_t cha;
 nchannel = ncha;
 for(int i=0; i<nchannel; i++)
 {
  cha.position[0] = val[i][0];
  cha.position[1] = val[i][1];
  cha.adc         = val[i][2];
 
  channel.push_back(cha);
 }

 double stripcluster[Spar],err[Spar];

 stripcluster[0] = (endpoint[1] + endpoint[0])/2;
 stripcluster[1] = (endpoint[1] - endpoint[0])/2;
 stripcluster[2] = 100.;

 LevenbergMarquardt<FitStripCluster> theLevenbergMarquardt;
 int nstep;
 stripcluster[Spar] =
  theLevenbergMarquardt.minimize(this, Spar, stripcluster, err, &nstep);

 double cluster[Npar+1];

 cluster[0] = stripcluster[0]; // center-x
 cluster[1] = 0.;              // center-y
 cluster[2] = stripcluster[1] * geom.pitch[0]; // half-length (cm)!

 cluster[4] = stripcluster[2]; // dE/dx
 cluster[Npar] = stripcluster[Spar];

 Transformations  theTransformations;

 // Try both direction, by flipping cluster direction / angle
 for(int entry=0; entry<2; entry++)
 {
  cluster[3] = entry * M_PI; // angle

  track[entry][Npar] = cluster[Npar];
  /*
  fprintf(stderr," %.3f %.3f  %.3f  %.3f\n",
		  cluster[0]-cluster[2]/geom.pitch[0],
		  cluster[0]+cluster[2]/geom.pitch[0],cluster[3],cluster[4]);
		  */
  theTransformations.transformClusterToTrack(cluster,track[entry]);
 }
}

