#include "Utilities/Configuration/interface/Architecture.h"

#include "TrackerReco/ClusterShaper/interface/ProcessTracker.h"

#include "TrackerReco/ClusterShaper/interface/ClusterMorphology.h"
#include "TrackerReco/ClusterShaper/interface/FitStripCluster.h"
#include "TrackerReco/ClusterShaper/interface/FitClusterPosition.h"
#include "TrackerReco/ClusterShaper/interface/FitClusterParameters.h"
#include "TrackerReco/ClusterShaper/interface/ProcessSimHit.h"

#include "TrackerReco/ClusterShaper/interface/EstimatePrimaryVertex.h"

#include "TrackerReco/ClusterShaper/interface/Transformations.h"

#include <iostream>

geom_t geom;

#define Ncol 8

#define Black   0
#define Red     1
#define Green   2
#define Yellow  3
#define Blue    4
#define Magenta 5
#define Cyan    6
#define White   7

static int color[Ncol] = {Black,Blue,Cyan,Green,Yellow,Magenta,Red,White};

extern TNtuple *ntuple;

/******************************************************************************/
ProcessTracker::ProcessTracker()
{
	/*
 outfile = new TFile("out/result.root","RECREATE");
 outfile->cd();
 z0dist = new TNtuple("z0dist","z0dist","z0");
 */
}

/******************************************************************************/
ProcessTracker::~ProcessTracker()
{
	/*
 outfile->cd();
 z0dist->Write("z0dist");
 outfile->Write();
 outfile->Close();
 */
}

/*****************************************************************************/
void ProcessTracker::plotCluster(int ncha, int val[][3])
{
 // Find container
 int extremum[2][2];
 for(int i=0; i<ncha; i++)
  for(int k=0; k<2; k++)
  {
   if(val[i][k] < extremum[k][0] || i==0) extremum[k][0] = val[i][k];
   if(val[i][k] > extremum[k][1] || i==0) extremum[k][1] = val[i][k];
  }

 cerr << "  [ClusterShaper]  " 
      << "(" << extremum[0][0] << "-" << extremum[0][1] << ") " 
      << "(" << extremum[1][0] << "-" << extremum[1][1] << ")" << endl;

 int x[2];

 for(x[1] = extremum[1][1]; x[1] >= extremum[1][0]; x[1]--)
 {
  fprintf(stderr,"   ");

  for(x[0] = extremum[0][0]; x[0] <= extremum[0][1]; x[0]++)
  {
   int adc = -1;

   for(int i=0; i<ncha; i++)
    for(int k=0; k<2; k++)
     if(val[i][0] == x[0] && val[i][1] == x[1])
      adc = val[i][2];

   if(adc >= 0)
   {
    int col = adc / 43 + 1; // color used Blue - Red
    fprintf(stderr, "%c[4%dm", 0x1B, color[col]);
    fprintf(stderr, "%c[3%dm", 0x1B, color[col]);
    fprintf(stderr, "%c[1m", 0x1B);
    fprintf(stderr,"%3d",adc);
    fprintf(stderr, "%c[m", 0x1B);
   }
   else fprintf(stderr,"   ");
  }
  fprintf(stderr,"\n");
 }
}

/******************************************************************************/
int ProcessTracker::getLayer(float z, float r)
{
 int layer = -1;

 if(fabs(z) < 30)
 {
  if(r < 5)          layer = 0;
  if(r > 6 && r < 8) layer = 1;
  if(r > 8         ) layer = 2;
 }

 if(fabs(z) > 30 && fabs(z) < 40) layer = 3;
 if(fabs(z) > 40)                 layer = 4;

 return(layer);
}

/******************************************************************************/
void ProcessTracker::showResult
  (const RecHit *recHit, int ncha, int minsize, int indep, double dmax,
   double rectrack[][Npar+1], double simtrack[Npar+1], int size[2])
{
	/*
 if(geom.pixel && geom.barrel && (minsize>=2 || indep==1))
  for(int i=0; i<2; i++)
  {
   float result = rectrack[i][3];
   z0dist->Fill(&result);
  }
  */

 // ROOT
 for(int i=0; i<2; i++)
 {
  float result[5 + Npar+1 + 1 + 4+1 + 2 + 1 + 1];

  int k = 0;

  result[k++] = recHit->globalPosition().z();
  result[k++] = recHit->globalPosition().perp();
  result[k++] = (float)getLayer(result[0],result[1]);
  result[k++] = ncha;
  result[k++] = minsize;

  for(int j=0; j<Npar+1; j++)
   result[k++] = rectrack[i][j];
  result[k++] = dmax;

  for(int j=0; j<4; j++)
   result[k++] = simtrack[j];
  result[k++] = simtrack[Npar];

  result[k++] = geom.barrel;
  result[k++] = geom.pixel ;

  result[k++] = geom.rot[2][0];

  result[k++] = indep;

  result[k++] = geom.event;

  result[k++] = geom.vertexz;

  result[k++] = size[0];
  result[k++] = size[1];

  ntuple->Fill(result);
 }

 // Fill zvertex
 for(int i=0; i<2; i++)
 if(geom.pixel && geom.barrel && size[1]>=2)
 {
  zvertex_t z; 

  z.z  = recHit->globalPosition().z();
  z.z0 = rectrack[i][3];

  zvertex.push_back(z);
  nzvertex++;
 }

 if(debug())
 {
  fprintf(stderr,
   "  [ClusterShaper]  she=%5.2f phi=%.2f qk=%6.3f z0=%5.1f type=%.0f\n",
    simtrack[0], simtrack[1], simtrack[2],
    simtrack[3], simtrack[Npar]);

  for(int i=0; i<2; i++)
   fprintf(stderr,
    "  [ClusterShaper]  she=%5.2f phi=%.2f qk=%6.3f z0=%5.1f dEdx=%4.2f chi2=%.1e\n",
    rectrack[i][0], rectrack[i][1], rectrack[i][2],
    rectrack[i][3], fabs(rectrack[i][4])*1e-3, rectrack[i][5]);

  cerr << "  [ClusterShaper]  cluster done" << endl;

  while(getchar()==0);
 }
}

/******************************************************************************/
void ProcessTracker::processStripRecHit
  (const RecHit *recHit, const StripDet* stripDet)
{
 double simtrack[Npar+1];
 ProcessSimHit theProcessSimHit;
 theProcessSimHit.associate(recHit, 0, stripDet, simtrack);

 int ncha = 0, val[1000][3];
 RecHit::ChannelContainer channels=recHit->channels();
 for(RecHit::ChannelIterator channel =channels.begin();
                             channel!=channels.end(); channel++)
 {
   val[ncha][0] = *channel;
   val[ncha][1] = 0;

   val[ncha][2] = stripDet->readout().channelAdc(*channel);
   ncha++;
 }

 // !!!!
 if(ncha >= 2 /*&& simtrack[Npar] == 2.*/)
 {
  if(debug())
  {
    cerr << "  [ClusterShaper]  plotting cluster" << endl;
    plotCluster(ncha,val);
  }
 
  int extremum[2] = {100000,-100};
  for(int i=0; i<ncha; i++)
  {
   if(val[i][0] > extremum[1]) extremum[1] = val[i][0];
   if(val[i][0] < extremum[0]) extremum[0] = val[i][0];
  }

  double endpoint[2];
 
  endpoint[0] = extremum[0] + 0.5 - 0.1; // !!!!
  endpoint[1] = extremum[1] + 0.5 + 0.1;
 
  if(debug()) cerr << "  [ClusterShaper]  fit strip-cluster" << endl;
  double rectrack[2][Npar+1];
  FitStripCluster theFitStripCluster;
  theFitStripCluster.fit(ncha,val, endpoint, rectrack);

 /*
 double cluster[Npar+1], rectrack[2][Npar+1];
 Transformations theTransformations;
 theTransformations.transformEndpointToCluster(endpoint, cluster);

 for(int entry=0; entry<2; entry++)
 {
  cluster[2] += entry * M_PI;
  theTransformations.transformClusterToTrack(cluster, rectrack[entry]);
 }
 */

  int minsize = 0;
  int indep   = 0;
  double dmax = 0.;
  int size[2] = {ncha,0};
 
  showResult(recHit, ncha,minsize,indep,dmax, rectrack,simtrack, size);
 }
}

/******************************************************************************/
void ProcessTracker::processPixelRecHit
  (const RecHit *recHit, const PixelDet* pixelDet)
{
 double simtrack[Npar+1];
 ProcessSimHit theProcessSimHit;
 theProcessSimHit.associate(recHit, pixelDet, 0, simtrack);

 int ncha = 0, val[1000][3], minsize, indep, edge = 0;
 RecHit::ChannelContainer channels=recHit->channels();
 for(RecHit::ChannelIterator channel =channels.begin();
                             channel!=channels.end(); channel++)
 {
   pair<int,int> pos = PixelDigi::channelToPixel(*channel);
   val[ncha][0] = pos.first;
   val[ncha][1] = pos.second;

   // Check if at the edge
   if(pos.first  == 0 || pos.first  == geom.dimension[0]-1 ||
      pos.second == 0 || pos.second == geom.dimension[1]-1)
    edge = 1;

   val[ncha][2] = pixelDet->readout().channelAdc(*channel);
   ncha++;
 }

 if(ncha >= 2 && !edge)
 {
  if(debug())
  {
   cerr << "  [ClusterShaper]  plotting cluster" << endl;
   plotCluster(ncha,val);
  }

  double dmax;
  ClusterMorphology theClusterMorphology;
  theClusterMorphology.pca(ncha,val, &dmax);
  if(debug()) cerr << "  [ClusterShaper]  dmax = " << dmax << endl;
 
  double endpoint[2][2];
  int size[2];
 
  if(debug()) cerr << "  [ClusterShaper]  fit cluster position" << endl;
  FitClusterPosition theFitClusterPosition;
  theFitClusterPosition.fit(ncha,val, endpoint, &minsize,&indep, size);
 
  if(debug())
   fprintf(stderr,"  [ClusterShaper]  endpoint: %.2f %.2f -> %.2f %.2f\n",
                 endpoint[0][0],endpoint[0][1],
                 endpoint[1][0],endpoint[1][1]);
 
  if(debug()) cerr << "  [ClusterShaper]  fit cluster parameters" << endl;
  double rectrack[2][Npar+1];
  FitClusterParameters theFitClusterParameters;
  theFitClusterParameters.fit(ncha,val, endpoint, rectrack);

  showResult(recHit, ncha,minsize,indep, dmax, rectrack,simtrack, size);
 }
}

/******************************************************************************/
void ProcessTracker::processDetUnit
  (const DetUnit* detUnit, Module module, Part part, geom_t *geom)
{
 geom->barrel = (part   == barrel);
 geom->pixel  = (module == pixel );

 // Thickness
 double thickZ = detUnit->surface().bounds().thickness();
 geom->thickness = thickZ;
  
 // Drift direction
 for(int k=0; k<2; k++) geom->shift[k] = 0.;
 LocalVector dir = detUnit->driftDirection(Local3DPoint(0.,0.,0.));
 geom->shift[0] = dir.x()/dir.z() * thickZ;
 geom->shift[1] = dir.y()/dir.z() * thickZ;
  
 // Global
 geom->pos[0]   =  detUnit->toGlobal( Local3DPoint(0.,0.,0.) ).x();
 geom->pos[1]   =  detUnit->toGlobal( Local3DPoint(0.,0.,0.) ).y();
 geom->pos[2]   =  detUnit->toGlobal( Local3DPoint(0.,0.,0.) ).z();

 // Rotation
 geom->rot[0][0] = detUnit->toGlobal( LocalVector(1.,0.,0.) ).x();
 geom->rot[1][0] = detUnit->toGlobal( LocalVector(1.,0.,0.) ).y();
 geom->rot[2][0] = detUnit->toGlobal( LocalVector(1.,0.,0.) ).z();

 geom->rot[0][1] = detUnit->toGlobal( LocalVector(0.,1.,0.) ).x();
 geom->rot[1][1] = detUnit->toGlobal( LocalVector(0.,1.,0.) ).y();
 geom->rot[2][1] = detUnit->toGlobal( LocalVector(0.,1.,0.) ).z();

 geom->rot[0][2] = detUnit->toGlobal( LocalVector(0.,0.,1.) ).x();
 geom->rot[1][2] = detUnit->toGlobal( LocalVector(0.,0.,1.) ).y();
 geom->rot[2][2] = detUnit->toGlobal( LocalVector(0.,0.,1.) ).z();

 // Flipped
 if(part == barrel)
 {
  float tmp1 = detUnit->toGlobal( Local3DPoint(0.,0.,0.) ).perp();
  float tmp2 = detUnit->toGlobal( Local3DPoint(0.,0.,1.) ).perp();
  if (tmp2 <tmp1) geom->flipped = 1; 
             else geom->flipped = 0;
 }
 else
 { 
  if(geom->rot[2][2] * geom->pos[2] > 0) geom->flipped = 0;
                                    else geom->flipped = 1;
 }

 // RecHits
 vector<RecHit> recHits = detUnit->recHits();

 const PixelDet* pixelDet = 0;
 const StripDet* stripDet = 0;

 if(module == pixel)
 { // pixel
  pixelDet = dynamic_cast<const PixelDet*>(detUnit);

  // Dimension, pitch
  geom->center[0] =
	  pixelDet->specificTopology().pixel(Local3DPoint(0,0,0)).first;
  geom->center[1] =
	  pixelDet->specificTopology().pixel(Local3DPoint(0,0,0)).second;

  geom->dimension[0] = pixelDet->specificTopology().nrows();
  geom->dimension[1] = pixelDet->specificTopology().ncolumns();

  geom->pitch[0] = pixelDet->specificTopology().pitch().first;
  geom->pitch[1] = pixelDet->specificTopology().pitch().second;
  geom->pitch[2] = geom->thickness;                     // thickness in z ??
 }
 else
 { // silicon
  stripDet = dynamic_cast<const StripDet*>(detUnit);

  // Dimension, pitch
  geom->center[0] = stripDet->specificTopology().strip(Local3DPoint(0,0,0));
  geom->center[1] = 0.;

  geom->dimension[0] = stripDet->specificTopology().nstrips();
  geom->dimension[1] = 0;

  geom->pitch[0] = stripDet->specificTopology().pitch();
  geom->pitch[1] = stripDet->specificTopology().stripLength(); // !!
 }

 for(int k=0; k<2; k++)
  geom->shift[k] /= geom->pitch[k];  // shift will be in units!

 if(debug())
  cerr << "  [ClusterShaper]"
       << " rechits=" << detUnit->recHits().size() << "/"
                      << geom->center[0]
       << " pos=" << geom->pos[0]
       << " " << geom->pos[1]
       << " " << geom->pos[2]
       << " shift=" << geom->shift[0]
       << endl;

 for(vector<RecHit>::const_iterator recHit =recHits.begin();
                                    recHit!=recHits.end(); recHit++)
  if(recHit->isValid())
  {
   if(debug())
    cerr << "------------------------------------------------------------------------------" << endl;

   if(active())
   if(module == pixel) processPixelRecHit(&(*recHit), pixelDet);
                  else processStripRecHit(&(*recHit), stripDet);
  }
}

/******************************************************************************/
void ProcessTracker::doAll()
{
 const CmsTracker::LayerContainer& all = FullTracker::instance()->allLayers();
 CmsTracker::LayerIterator ilayer;

 cerr << "  [ClusterShaper] debug = " << debug();

 zvertex.clear();

 Module oldmodule = silicon;
 Part   oldpart   = forward;

 nzvertex=0;

 for(ilayer = all.begin(); ilayer != all.end(); ilayer++)
 {
  if(ilayer->module() != oldmodule || ilayer->part() != oldpart)
  {
   cerr << "\n  [ClusterShaper] " << ilayer->module() << ilayer->part();
   oldmodule = ilayer->module(); oldpart = ilayer->part();
  }
  else
   cerr << ".";

   if( (processPixels() && ilayer->module() == pixel  ) ||
       (processStrips() && ilayer->module() == silicon) )
   {
    // Detunits
    Det::DetUnitContainer theDetUnits = ilayer->detUnits();
    for(Det::DetUnitContainer::iterator detUnit  = theDetUnits.begin();
                                        detUnit != theDetUnits.end(); detUnit++)
      processDetUnit(*detUnit, ilayer->module(), ilayer->part(), &geom);
   }
 }
 cerr << endl;

 EstimatePrimaryVertex theEstimatePrimaryVertex;
 theEstimatePrimaryVertex.fit(nzvertex,zvertex);
}

