#include "Utilities/Configuration/interface/Architecture.h"

#include "TrackerReco/ClusterShaper/interface/ChannelsTouched.h"

#include <cmath>

#include <iostream>

template <class T> inline const T& Max (const T& x, const T& y) 
{ return (x > y ? x : y); } 

template <class T> inline const T& Min (const T& x, const T& y) 
{ return (x < y ? x : y); } 

extern geom_t geom;

/*****************************************************************************/
ChannelsTouched::ChannelsTouched() {}

/*****************************************************************************/
ChannelsTouched::~ChannelsTouched() {}

/*****************************************************************************/
void ChannelsTouched::invertCross(cross_t *a, cross_t *b)
{
 cross_t c;

 for(int k=0; k<2; k++)
 {
  c.pos[k] = a->pos[k]; a->pos[k] = b->pos[k]; b->pos[k] = c.pos[k];
 }

 c.edg = a->edg; a->edg = b->edg; b->edg = c.edg;
}

/*****************************************************************************/
void ChannelsTouched::moveCross(cross_t *a, cross_t *b)
{
 for(int k=0; k<2; k++)
 {
  a->pos[k] = b->pos[k]; 
 }

 a->edg = b->edg;
}

/*****************************************************************************/
void ChannelsTouched::sortVectors1(int *n,cross_t cross[],int dir)
{
 int change;

 do
 {
  change=0;

  for(int i=0; i<*n-1; i++)
  {
   // Reverse order
   if(cross[i].pos[dir] > cross[i+1].pos[dir])
   {
    invertCross(&cross[i],&cross[i+1]);
    change++;
   }

   // If same, remove one of them
   if(cross[i].pos[dir] == cross[i+1].pos[dir])
   {
    for(int j=i; j<*n-1; j++)
     moveCross(&cross[j],&cross[j+1]);

    i--; (*n)--;
    change++;
   }
  }
 }
 while(change);
}

/*****************************************************************************/
void ChannelsTouched::lookForCrossing(double point[][2],double dpos[],
                     int *n,vector<cross_t>* cross,int dir)
{
 int e[2];

 e[0] = (int)Min(point[0][dir],point[1][dir]) + 1;
 e[1] = (int)Max(point[0][dir],point[1][dir])    ;

 // Bring back
 for(int i=0; i<2; i++)
 {
  if(e[i] < 0                   ) e[i] = 0;
  if(e[i] > geom.dimension[dir]) e[i] = geom.dimension[dir];
 }

 for(int i=e[0]; i<=e[1]; i++) // Go from minimum to maximum
 {
  cross_t cr;

  cr.pos[dir]   = i;
  cr.pos[1-dir] = point[0][1-dir] +
                dpos[1-dir]*(i - point[0][dir])/dpos[dir];

  cr.edg = dir;

  cross->push_back(cr);

  (*n)++;
 }
}

/*****************************************************************************/
void ChannelsTouched::calculatePositionAndLength(pixel_t *pixel, cross_t c[],
       int dir, double dpos[], double center[], double length)
{
 // Position
 for(int k=0; k<2; k++)
  pixel->position[k] = (int)((c[1].pos[k] + c[0].pos[k])/2);

 pixel->nc       = 0;
 pixel->endpoint = 0;

 // Take cross/end points
 for(int j=0; j<2; j++)
 {
  int n = c[j].edg;  

  if(n >= 0)
  { // Normal, radius 
   pixel->normal[pixel->nc][  n] = (c[1-j].pos[n] > c[j].pos[n] ? 1. : -1.);
   pixel->normal[pixel->nc][1-n] = 0.;

   pixel->dx[pixel->nc][0] = (c[j].pos[0] - center[0]) * geom.pitch[0];
   pixel->dx[pixel->nc][1] = (c[j].pos[1] - center[1]) * geom.pitch[1];

   (pixel->nc)++;
  }
  else // Endpoint
   (pixel->endpoint)++;

 }

 // Length
 if(dpos[dir] != 0.)
  pixel->length = fabs((c[1].pos[dir] - c[0].pos[dir])/dpos[dir]) * length;
 else
  pixel->length = length;
}

/*****************************************************************************/
void ChannelsTouched::calculateTotalLength(double point[][2], double *length2d, double *length3d)
{
 double dpos[2];

 *length2d = 0.; // Total 2d length!!
 *length3d = 0.; // Total 3d length!!

 for(int k=0; k<2; k++)
 {
  dpos[k] = (point[1][k] - point[0][k]);
  *length2d += sqr(dpos[k] * geom.pitch[k]);

  dpos[k] = (point[1][k] - point[0][k]) +
            (!geom.flipped ? 1 : -1) * (-geom.shift[k]); // undo Lorentz
  *length3d += sqr(dpos[k] * geom.pitch[k]);
 }

 *length3d += sqr(geom.thickness); // thickness

 *length2d = sqrt(*length2d);
 *length3d = sqrt(*length3d);
}

/*****************************************************************************/
// Find positions and lengths for channels belonging to hit
int ChannelsTouched::getPath(double point[][2], vector<pixel_t>* pixel)
{
 int n;
 double dpos[2], length2d,length3d, center[2];

 // Calculate total path-length and center
 calculateTotalLength(point, &length2d,&length3d);
//fprintf(stderr," length 2d= %.1fum 3d=%.1fum\n",length2d*1e+4,length3d*1e+4);

 for(int k=0; k<2; k++)
  dpos[k] = (point[1][k] - point[0][k]);

 for(int k=0; k<2; k++)
 {
  if(point[1][k] != point[0][k])
   center[k] = (point[1][k] + point[0][k])/2;
  else
   center[k] = point[1][k];
 }

 // Collect crossing points
 // cross.edg = -1 means endpoint
 // cross.edg =  0 means crossing vertical
 // cross.edg =  1 mean  crossing horizontal

 vector<cross_t> cross;
 cross.reserve(10);
 n=0;
 for(int i=0; i<2; i++) // starting and ending points
 {
  cross_t cr;
  for(int k=0; k<2; k++)
   cr.pos[k] = point[i][k];
  cr.edg = -1;

  cross.push_back(cr); n++;
 }

 int dir;

 for(dir=0; dir<2; dir++) // x and y directions
  if(dpos[dir] != 0.)
   lookForCrossing(point,dpos, &n,&cross,dir);

 // Sort to increasing order in [dir]
 if(fabs(dpos[0]) > fabs(dpos[1])) dir=0; else dir=1;
 sortVectors1(&n,&cross[0],dir);

 n--;

 // Calculate position and path length
 for(int i=0; i<n; i++)
 {
  cross_t c[2];
  moveCross(&c[0],&cross[i  ]);
  moveCross(&c[1],&cross[i+1]);

  pixel_t pix;
  calculatePositionAndLength(&pix, c, dir, dpos,center,length3d);
  pixel->push_back(pix);
 }

 return(n);
}

