#ifndef _LEVENBERG_MARQUARDT_H_
#define _LEVENBERG_MARQUARDT_H_

#include "Header.h"

#include "TrackerReco/ClusterShaper/interface/Matrix.h"

#include <vector>

template <class T>
class LevenbergMarquardt
{
 public:  
   LevenbergMarquardt();
   ~LevenbergMarquardt();
   double minimize
    (T *theModel, int npar, double par[], double err[], int *nstep);

 private:
   T *theModel;
   void gaussJordan
     (int dim, Matrix<double>* matrix, double vec[]);
   void invertMatrix
     (int dim, Matrix<double>* matrix);
   double moveStep
     (int dim, double old, double par[], double *lambda, double err[]);
};


#include "Utilities/Configuration/interface/Architecture.h"

#include <cmath>
#include <iostream>

#define Lambda_ini 1e-3
#define Lambda_max 1e+3

#define Dchi2_max -1e-2
#define Steps_max  100

/*****************************************************************************/
template <class T>
LevenbergMarquardt<T>::LevenbergMarquardt() {}

/*****************************************************************************/
template <class T>
LevenbergMarquardt<T>::~LevenbergMarquardt() {}

/*****************************************************************************/
template <class T>
void LevenbergMarquardt<T>::gaussJordan
  (int dim, Matrix<double>* matrix, double vec[])
{
 /* Elimination */
 for(int i=0; i<dim-1; i++)
  for(int j=i+1; j<dim; j++)
  if((*matrix)(j,i)!=0.)
  {
   double ratio=(*matrix)(j,i)/(*matrix)(i,i); 

   (*matrix)(j,i)=0.;

   for(int k=i+1; k<dim; k++)
    (*matrix)(j,k) -= ratio*(*matrix)(i,k);

   vec[j] -= ratio*vec[i];
  }
 
 /* Back-substitution */
 for(int i=dim-1; i>=0; i--)
 {
  double others=0.;
  for(int j=i+1; j<dim; j++) others+=(*matrix)(i,j)*vec[j];
  
  vec[i]=(vec[i]-others)/(*matrix)(i,i);  
 }
}

/*****************************************************************************/
// Check public/CmsHi/dEdx/save/Jun30_23:19/nr_invert.c
template <class T>
void LevenbergMarquardt<T>::invertMatrix
  (int dim, Matrix<double>* matrix)
{
 Matrix<double> m(dim,dim), inverse(dim,dim);

 /* Invert by solving matrix equations for every column */
 for(int i=0; i<dim; i++)
 {
  for(int j=0; j<dim; j++)
  for(int k=0; k<dim; k++)
   m(j,k) = (*matrix)(j,k);

  for(int j=0; j<dim; j++) inverse(i,j) = (i==j ? 1. : 0.);
  gaussJordan(dim,&m, &inverse(i)[0]);
 }

 for(int i=0; i<dim; i++)
 for(int j=0; j<dim; j++)
  (*matrix)(i,j) = inverse(j,i);
}

/*****************************************************************************/
template <class T>
double LevenbergMarquardt<T>::moveStep
  (int dim, double old, double par[], double *lambda, double err[])
{
 vector<double> b(dim),beta(dim);
 double val;
 Matrix<double> a(dim,dim), alpha(dim,dim); 
 int i,j, ok=0;

 // Get derivatives
 theModel->getDerivatives(par, &b[0],&a);

 /* If not dependent, repair matrix */
 for(i=0; i<dim; i++)
  if(b[i]==0. || a(i,i)==0.)
  {
   b[i]=0.;

   for(j=0; j<dim; j++)
   if(i!=j)
   { a(i,j) = 0.; a(j,i) = 0.; }

   a(i,i) = 1.;
  }

 do
 {
  /* Set beta and alpha */
  for(i=0; i<dim; i++)
  {
   beta[i] = b[i];

   for(j=0; j<dim; j++)
    if(i!=j) alpha(i,j) = a(i,j);
        else alpha(i,j) = a(i,j) * (1 + *lambda);
  }

  if(*lambda != 0.)
  {
   /* Solve */
   gaussJordan(dim, &alpha,&beta[0]); 

   /* Update */
   for(i=0; i<dim; i++)
    beta[i] += par[i];
 
   theModel->getFunction(&beta[0], &val);
 
   if(val<=old)
   {
    ok=1; *lambda /= 10.;
    for(i=0; i<dim; i++) par[i]=beta[i];
   }
   else
    *lambda *= 10;
  }
  else
  {
   ok = 1;

   theModel->getFunction(&par[0], &val);

   invertMatrix(dim,&alpha);

   for(i=0; i<dim; i++) err[i] = sqrt(alpha(i,i));
  }
 }
 while(!ok && *lambda < Lambda_max);

 return(val);
}

/*****************************************************************************/
template <class T>
double LevenbergMarquardt<T>::minimize
 (T *model, int npar, double par[], double err[], int *nstep)
{
 double old,val, change, lambda;

 *nstep = 0;
 theModel = model;

 theModel->getFunction(par, &old);
 if(debug() || npar==1)
  fprintf(stderr,"  [ClusterShaper]  first call = %.2f\n",old);
 lambda = Lambda_ini;

 do
 {
  val = moveStep(npar,old, par, &lambda, err);
  change = val-old; 

  if(debug() || npar==1)
   fprintf(stderr,"  [ClusterShaper]   change = %.2f | %.0e\n",change,lambda);

  old = val; (*nstep)++;
 }
 while((change < Dchi2_max || change > 0) &&
       *nstep < Steps_max && lambda < Lambda_max);

 lambda = 0.;
 val = moveStep(npar,old, par, &lambda, err);

 if(debug() || npar==1)
  fprintf(stderr,"  [ClusterShaper]  last call = %.2f [step=%d lambda=%.1e]\n",
                 val,*nstep, lambda);

 return(val);
}


#endif
